/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Diagnostic Function Manager API Definition
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file DFMProtocol.h
    \brief The Diagnostic Function Manager (DFM) protocol is embedded within the ERI protocol. Its purpose is
           to allow application services to be started remotely. Typically it is used to start diagnostic like
           services, hence the DFM name. The services supported, and their definition, is dependant upon the
           application. DFM only describes the mechanism through which these services can be accessed.
           
           The application needs to supply the protocol with access to the diagnostic function manager before
           DFM function can be used. This is achieved through the use of a function pointer. The manager
           function is given execution cotnrol when the ERI protocol detects a DFM request. One of the
           parameters of the diagnostic function manager call is a reply callback that allows the manager to
           provide the protocol with an appropriate, context sensitive relpy.
           
           The DFM request will reply with a failure, specifically \ref DFM_DR_NOT_A_FUNC, to indicate that
           the function is not supported when a diagnostic function manager callback has not been supplied to
           the registered Protocol archive.
           
           See RegisterMotoTronProtocolArchive() and the \ref S_RegisterProtocolArchiveInfo data structure. */

#ifndef __DFMPROTOCOL_H
#define __DFMPROTOCOL_H

/*---- INCLUDE FILES --------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <ProtocolAccess.h>

/*---- DEFINES --------------------------------------------------------------------------------------------*/

/*---- TYPEDEF --------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! Protocol defined general replys for the DFM */
typedef enum
{
/*! General reply that indicates that the request failed */
    DFM_GR_REQ_FAILED    = 0x01,    
/*! General reply that indicates that the requested test failed */
    DFM_GR_TST_FAILED    = 0x02,
/*! General reply indicating that a requested operation is in progress */
    DFM_GR_IN_PROGRES    = 0x10,
/*! General reply indicating that a requested operation is in progress and that the detailed reply indicates
   the progress of that request as a percentage. */
    DFM_GR_PCENT_COMP    = 0x11,
/*! General reply that indicates that a request was successful */
    DFM_GR_SUCCESSFUL    = 0x20,
/*! General reply that indicates some data was returned */
    DFM_GR_DATA_RETURNED = 0x40,
/*! General reply. Indicates that the requested function failed to execute */
    DFM_GR_FUNC_FAILED   = 0x80
} E_DFMGenReply;

/*! Protocol defined detailed reply values for the DFM protocol. A reply is constructed from a general and
    detailed reply. The detailed reply is housed in the \b LSByte of a reply. The general reply is found in
    the MSByte of the reply. Use \ref DFM_DR_NO_INFO when a detailed replied is not supported by a request. */
typedef enum
{
/*! Detailed reply that indicates that the detailed reply does not contain any information */
    DFM_DR_NO_INFO               = 0x00,
/*! The /ref E_DFMGenReply resulted because the module had detected engine speed. Some functions require
    that there not be any engine speed present before they can be employed */
    DFM_DR_ENGINE_SPD_PRSNT      = 0x01,
/*! The /ref E_DFMGenReply resulted because the request had insufficient privileges */
    DFM_DR_NO_PRIV               = 0x02,
/*! The /ref E_DFMGenReply resulted because there was an error in the request's supplied data */
    DFM_DR_DATA_ERR              = 0x03,
/*! The request failed because the diagnostic function manager was busy. It is probably already servicing
    too many functions. Often some functions will need to be aborted before others can proceed */
    DFM_DR_DFM_BUSY              = 0x04,
/*! The request failed because the diagnostic function manager does not support the requested function. This
    reply also results if the module does not support a diagnostic function manager. */
    DFM_DR_NOT_A_FUNC            = 0x05,
/*! The /ref E_DFMGenReply resulted because the request is already in progress. */
    DFM_DR_IN_PROGRES            = 0x06,
/*! The request failed because an invalid DFM command was supplied */
    DFM_DR_NOT_A_CMD             = 0x07,
/*! The request failed because the function was not active. Often returned when requesting the status of an
    inactive function. */
    DFM_DR_NOT_ACTIVE            = 0x08,
/*! The request failed because the module does not support the requested hardware */
    DFM_DR_NO_HWARE              = 0x09,
/*! The /ref E_DFMGenReply resulted because of a hardware fault */
    DFM_DR_HWARE_FAULT           = 0x0A,
/*! The request failed because it required the use of hardware that was already in use by the module or a
    separate DFM request */
    DFM_DR_HWARE_IN_USE          = 0x0B,
/*! The /ref E_DFMGenReply resulted because the module did not detect any engine speed. Some functions require
    that there engine speed present before they can be employed */
    DFM_DR_ENGINE_SPD_REQUIRED   = 0x0C
} E_DFMDetailReply;

/*! The DFM protocol provides a recomendation on the order of the detailed and general reply, which this
    structure facilitates */
typedef struct
{
    uint1 u1DetailedReply;
    uint1 u1GeneralReply;
} S_DFMReply;
   
typedef union         
{
    uint2 u2Reply;
    S_DFMReply Bytes;
} U_DiagFuncReply;

/*! The valid Diagnostic Function Manager (DFM) commands */
typedef enum 
{
/*! Initiates a DFM request */
    DFM_INITIATE_CMD    = 0x25,
/*! Requests an update */
    DFM_UPDATE_CMD      = 0x2A,
/*! Request the status */
    DFM_STATUS_CMD      = 0x33,    
/*! Abort a previously initiated request */
    DFM_ABORTHALT_CMD   = 0x3C
} E_DFMCommands;

/*! \brief Function pointer type for the reply to a Diagnostic Function Manager (DFM) request

 \p in_u2DFMReply represents the reply to the request and \p in_uCbackData is the context specific callback
 data that was supplied with the DFM request, see \ref DFMFuncPtr. */
typedef void (*DFMFuncReplyCBackPtr)(uint2 in_u2DFMReply, NativePtrSizedInt_U in_uCbackData);

/*! \brief Function pointer type for Diagnostic Function Manager (DFM) service requests.

 This function executes when a DFM request is processed by an appropriate protocol. The request is
 dispatched to this manager. The protocol will respond with a failure if the manager is not supported.
 This callback is configured when the archive is registered for use with RegisterMotoTronProtocolArchive().
 See the S_RegisterProtocolArchiveInfo::pfDiagnosticFunctionManager member.

 \p in_u1FuncCode describes the function code that the request is to affect. Its context is defined by the
 application. \p in_u2FuncData contains the data to be applied to the function. It is context is defined by
 the function code and the command. \ref E_DFMCommands describes the DFM command to apply to the function
 identified by \p in_u1FuncCode. \ref E_AccessLevel describes the access level of this request.
 \ref DFMFuncReplyCBackPtr describes the callback function that the application should execute inorder to reply
 to this DFM request. \p in_uDataToSendWithCback is context sensitive information that should be supplied when
 the \ref DFMFuncReplyCBackPtr reply function callback is executed. */
typedef void (*DFMFuncPtr)(uint1 in_u1FuncCode, uint2 in_u2FuncData, E_DFMCommands, E_AccessLevel, DFMFuncReplyCBackPtr, NativePtrSizedInt_U in_uDataToSendWithCback);

#pragma pack()
/*---- EXTERNALS ------------------------------------------------------------------------------------------*/

/*---- PROTOTYPES -----------------------------------------------------------------------------------------*/

#endif /* __DFMPROTOCOL_H */

/*---- END OF FILE ----------------------------------------------------------------------------------------*/

